#Requires AutoHotkey v2.0
; =========================================================
; NinjaTrader Tab Switcher — Screen Coordinates Core
; Hotkeys:
;   Alt+1..Alt+0       -> click saved point for Tab 1..10
;   Ctrl+Alt+1..0      -> save current cursor point for Tab 1..10 (auto-save)
;   Alt+C              -> preview ALL saved tabs (tooltips, no clicks)
;   Ctrl+Alt+Arrows    -> nudge last-used tab by ±1 px (Shift = ±5 px), auto-save
; =========================================================

; ---------- USER OPTIONS ----------
RestoreMouse := true            ; true = return cursor to original position after the click
ActivateNinjaTrader := false    ; true = focus NinjaTrader before clicking (still uses screen coords)
DoubleClickTabs := false        ; true = double click the saved point instead of single click
ClickDelayMs := 40              ; delay between double-clicks if enabled

; Work in absolute screen space (simple & reliable across monitors)
CoordMode("Mouse", "Screen")

global SETTINGS_PATH := A_ScriptDir "\BlackbayTabSwitcher.ini"
global TabSX := Map()   ; index -> Screen X
global TabSY := Map()   ; index -> Screen Y
global TabMon := Map()  ; index -> Monitor index (info only)
global LastIndex := 0   ; last tab index used (for nudging)

LoadSettings()

; -------- Click hotkeys (Alt+1..Alt+0)
!1::ClickTab(1)
!2::ClickTab(2)
!3::ClickTab(3)
!4::ClickTab(4)
!5::ClickTab(5)
!6::ClickTab(6)
!7::ClickTab(7)
!8::ClickTab(8)
!9::ClickTab(9)
!0::ClickTab(10)

; -------- Save hotkeys (Ctrl+Alt+1..0)
^!1::SaveTabPoint(1)
^!2::SaveTabPoint(2)
^!3::SaveTabPoint(3)
^!4::SaveTabPoint(4)
^!5::SaveTabPoint(5)
^!6::SaveTabPoint(6)
^!7::SaveTabPoint(7)
^!8::SaveTabPoint(8)
^!9::SaveTabPoint(9)
^!0::SaveTabPoint(10)

; -------- Preview ALL saved tabs (tooltips only)
!c::PreviewAllTabs()

; -------- Nudge last-used tab point (Shift = 5 px)
^!Up::    NudgeLast(0, -1)
^!Down::  NudgeLast(0, +1)
^!Left::  NudgeLast(-1, 0)
^!Right:: NudgeLast(+1, 0)

+^!Up::    NudgeLast(0, -5)
+^!Down::  NudgeLast(0, +5)
+^!Left::  NudgeLast(-5, 0)
+^!Right:: NudgeLast(+5, 0)

; --------------- FUNCTIONS ----------------

ClickTab(index) {
    global LastIndex, ActivateNinjaTrader, RestoreMouse, DoubleClickTabs, ClickDelayMs
    if !(TabSX.Has(index) && TabSY.Has(index)) {
        QuickTip("Tab #" index " not set. Use Ctrl+Alt+" index " to save a point.", , , 1500)
        return
    }
    LastIndex := index

    if (ActivateNinjaTrader) {
        h := WinExist("ahk_exe NinjaTrader.exe")
        if (h) {
            WinActivate("ahk_id " h)
            Sleep(40)
        }
    }

    ox := 0, oy := 0
    if (RestoreMouse) {
        MouseGetPos(&ox, &oy)
    }

    if (DoubleClickTabs) {
        Click(TabSX[index], TabSY[index])
        Sleep(ClickDelayMs)
        Click(TabSX[index], TabSY[index])
    } else {
        Click(TabSX[index], TabSY[index])
    }

    if (RestoreMouse) {
        MouseMove(ox, oy, 0)
    }
}

SaveTabPoint(index) {
    global LastIndex
    MouseGetPos(&sx, &sy)
    TabSX[index] := sx
    TabSY[index] := sy
    TabMon[index] := GetMonitorIndexFromPoint(sx, sy)
    SaveTab(index)
    LastIndex := index
    QuickTip("Saved Tab #" index "  X=" sx "  Y=" sy "  (Mon " (TabMon[index] ? TabMon[index] : "?") ")", , , 1000)
}

PreviewAllTabs() {
    shown := false
    loop 10 {
        idx := A_Index
        if (TabSX.Has(idx) && TabSY.Has(idx)) {
            ToolTip("↘ Tab #" idx, TabSX[idx] + 10, TabSY[idx] + 10)
            shown := true
            Sleep(700)  ; show each marker ~0.7s
        }
    }
    if (shown) {
        SetTimer(() => ToolTip(), -10)  ; clear tooltip
    } else {
        ToolTip("No tabs calibrated yet. Use Ctrl+Alt+1..0 to save.")
        SetTimer(() => ToolTip(), -1200)
    }
}

NudgeLast(dx, dy) {
    global LastIndex
    if (LastIndex = 0) {
        QuickTip("No tab selected yet. Click or save a tab first.", , , 900)
        return
    }
    if !(TabSX.Has(LastIndex) && TabSY.Has(LastIndex))
        return
    TabSX[LastIndex] := TabSX[LastIndex] + dx
    TabSY[LastIndex] := TabSY[LastIndex] + dy
    SaveTab(LastIndex)
    ToolTip("Nudged Tab #" LastIndex " → X=" TabSX[LastIndex] " Y=" TabSY[LastIndex]
        , TabSX[LastIndex] + 10, TabSY[LastIndex] + 10)
    SetTimer(() => ToolTip(), -600)
}

QuickTip(text, x:=unset, y:=unset, ms:=900) {
    if (IsSet(x) && IsSet(y)) {
        ToolTip(text, x, y)
    } else {
        ToolTip(text)
    }
    SetTimer(() => ToolTip(), -ms)
}

; ------------- Monitor helper (info only) -------------
GetMonitorIndexFromPoint(x, y) {
    try {
        count := MonitorGetCount()
        loop count {
            MonitorGet(A_Index, &L, &T, &R, &B)
            if (x >= L && x < R && y >= T && y < B)
                return A_Index
        }
    } catch {
        ; ignore
    }
    return 0
}

; ------------- INI I/O -------------
LoadSettings() {
    global TabSX, TabSY, TabMon, SETTINGS_PATH
    if !FileExist(SETTINGS_PATH)
        return
    try {
        loop 10 {
            k := A_Index
            xs := IniRead(SETTINGS_PATH, "PerTab", "TabSX" k, "")
            ys := IniRead(SETTINGS_PATH, "PerTab", "TabSY" k, "")
            mn := IniRead(SETTINGS_PATH, "PerTab", "TabMon" k, "")
            if (xs != "" && ys != "") {
                TabSX[k] := Integer(xs)
                TabSY[k] := Integer(ys)
            }
            if (mn != "") {
                TabMon[k] := Integer(mn)
            }
        }
    } catch {
        ; ignore read errors
    }
}

SaveTab(index) {
    global TabSX, TabSY, TabMon, SETTINGS_PATH
    try {
        IniWrite(TabSX.Has(index) ? TabSX[index] : "", SETTINGS_PATH, "PerTab", "TabSX" index)
        IniWrite(TabSY.Has(index) ? TabSY[index] : "", SETTINGS_PATH, "PerTab", "TabSY" index)
        IniWrite(TabMon.Has(index) ? TabMon[index] : "", SETTINGS_PATH, "PerTab", "TabMon" index)
    } catch {
        QuickTip("Failed to save Tab #" index " to INI.", , , 1200)
    }
}
